package by.avest.edoc.examples;

import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.Security;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Properties;
import java.util.TimeZone;

import by.avest.certstore.AvCertStoreProvider;
import by.avest.crypto.pkcs11.provider.AvestProvider;
import by.avest.crypto.pkcs11.provider.ProviderFactory;
import by.avest.edoc.client.AvDocException;
import by.avest.edoc.client.AvEDoc;
import by.avest.edoc.client.AvEList;
import by.avest.edoc.client.EVatService;
import by.avest.net.tls.AvTLSProvider;

/**
 * Скачивание счетов-фактур НДС с портала
 */
public class ListAndGet {

	public static void main(String[] args) {
		boolean isVerbose = false;

		AvestProvider prov = null;

		try {
			// Регистрация провайдера AvJceProv
			prov = ProviderFactory.addAvUniversalProvider();
			Security.addProvider(new AvTLSProvider());
			Security.addProvider(new AvCertStoreProvider());

			// Получение параметров командной строки
			if (args.length < 2) {
				printUsage("Ошибка: не указаны входные параметры");
				return;
			}

			String serviceUrl = args[0];
			String docFolderName = args[1];

			if (2 < args.length && args[2].equalsIgnoreCase("v")) {
				isVerbose = true;
			}

			// Создание экземпляра класса доступа к порталу
			EVatService service = null;

			try {
				service = new EVatService(serviceUrl, new KeyInteractiveSelector());

				// Авторизация, выбор ключа и ввода пароля выполняется
				// интерактивно
				// при помощи класса KeyInteractiveSelector
				service.login();

				// Подключение к автоматизированному сервису портала
				service.connect();

				// Чтение даты последнего запроса списка счетов-фактур на
				// портале
				Date fromDate;
				Properties properties = new Properties();
				File file = new File(docFolderName, "ListEDocuments.dat");
				if (!file.exists()) {
					file.createNewFile();
				}
				FileInputStream fileInput = new FileInputStream(file);
				try {
					properties.load(fileInput);
				} finally {
					fileInput.close();
				}

				String fromDateStr = properties.getProperty("fromDate");
				if (fromDateStr != null) {
					fromDate = string2Date(fromDateStr);
				} else {
					// Дата последнего запроса не сохранена в файл, запрашиваем
					// за
					// последние 365 дней
					Calendar cal = Calendar.getInstance();
					cal.add(Calendar.DATE, -365);
					fromDate = cal.getTime();
				}

				// Получение списка поступивщих счетов-фактур НДС
				AvEList list = service.getList(fromDate);
				// Проверка ЭЦП электронного документа
				boolean isValid = list.verify();
				if (isValid) {
					if (list.getCount() == 0) {
						System.out.println("На сервере нет ни одного нового счета-фактуры.");
					} else {
						System.out.println("Получены следующие счета-фактуры:");

						for (int i = 0; i < list.getCount(); i++) {
							String invoiceNum = (String) list.getItemAttribute(i, "document/number");

							// Получение электронного документа счет-фактуры по
							// номеру
							AvEDoc eDocXml = service.getEDoc(invoiceNum);

							// Проверка ЭЦП электронного документа
							isValid = edocVerify(eDocXml);
							if (isValid) {
								// Сохранение электронного документа в файл
								File docFile = new File(docFolderName, "invoice-" + invoiceNum + "-verified.xml");
								writeFile(docFile, eDocXml.getEncoded());

								// Сохранение счет-фактуры НДС в файл
								File issuanceFile = new File(docFolderName, "invoice-" + invoiceNum + ".xml");
								writeFile(issuanceFile, eDocXml.getDocument().getEncoded());

								System.out.println(
										"Счет-фактура с номером " + invoiceNum + " получена и сохранена в файл.");

							} else {
								// Сохранение электронного документа в файл
								File docFile = new File(docFolderName, "ERROR-invoice-" + invoiceNum + ".xml");
								writeFile(docFile, eDocXml.getEncoded());

								System.err.println("Ошибка: получена некорректная счет-фактура с номером \""
										+ invoiceNum + "\": " + eDocXml.getLastError().getMessage());
							}
						}
					}
				} else {
					System.err.println("Ошибка: получен некорректный список поступивших счетов-фактур");
				}

				// Сохранение даты получения списка счетов-фактур для следующего
				// запуска
				Date toDate = list.getToDate();
				properties.setProperty("fromDate", date2String(toDate));
				FileOutputStream out = new FileOutputStream(new File(docFolderName, "ListEDocuments.dat"));
				try {
					properties.store(out, "");
				} finally {
					out.close();
				}
			} catch (Exception e) {
				System.err.println("Ошибка: " + e.getMessage());
				if (isVerbose) {
					e.printStackTrace();
				}
				System.exit(1);
			} finally {
				if (service != null) {
					try {
						// Завершение работы с сервисом
						service.disconnect();

						// Завершение авторизованной сессии
						service.logout();
					} catch (IOException e) {
						// ignore
					}
				}
			}
		} finally {
			if (prov != null) {
				prov.close();
			}
		}
	}

	protected static final String OID_CN = "2.5.4.3";

	private static final String SERVER_DN_CN = "Автоматизированный сервис портала АИС УСФ";

	protected static boolean isServerSign(AvEDoc edoc, int i) throws IOException, AvDocException {
		String commonName = edoc.getSignProperty(i, OID_CN);

		return (SERVER_DN_CN.equals(commonName));
	}

	private static int findServerSign(AvEDoc edoc) throws IOException, AvDocException {
		int result = -1;

		int signCount = edoc.getSignCount();

		for (int i = 0; i < signCount; i++) {
			if (isServerSign(edoc, i)) {
				// Найдена подпись сервера
				result = i;
				break;
			}
		}

		return result;
	}

	private static boolean edocVerify(AvEDoc edoc) throws AvDocException, IOException {
		boolean isValid = false;

		// Количество подписей
		int signCount = edoc.getSignCount();
		if (signCount == 0) {
			System.out.println("Документ не содержит ЭЦП");
			return false;
		}

		// Найти подпись сервера
		int serverIndex = findServerSign(edoc);

		if (0 < serverIndex) {
			System.out.println("Проверка подписи сервера");

			isValid = edoc.verifySign(serverIndex);

			// Print verification result
			if (isValid) {
				String signdate = edoc.getSignProperty(serverIndex, AvEDoc.PROP_SIGN_DATE);
				System.out.println("ЭЦП проверена. Дата подписания: " + signdate);
			} else {
				System.out.println(
						"ЭЦП неверна. " + ((edoc.getLastError() != null) ? edoc.getLastError().getMessage() : "-"));
			}
		} else {
			System.out.println("Подпись сервера не найдена. Будет выполнена проверка всех подписей.");

			int count = edoc.getSignCount();

			for (int j = 0; j < count; j++) {
				// Информация о стороне подписавшей документ
				String cn = edoc.getSignProperty(j, "2.5.4.3");
				System.out.println("\tПроверка подписи " + cn);

				// Проверка ЭЦП электронного документа
				isValid = edoc.verifySign(j);

				if (!isValid) {
					break;
				}
			}

			// Print verification result
			if (isValid) {
				System.out.println("Все ЭЦП проверены успешно.");
			} else {
				System.out.println(
						"ЭЦП неверна. " + ((edoc.getLastError() != null) ? edoc.getLastError().getMessage() : "-"));
			}
		}

		return isValid;
	}

	/**
	 * Отображение справки по параметрам командной строки
	 */
	private static void printUsage(String message) {
		System.out.println("ListAndGet: " + message);
		System.out.println("Использование:");
		System.out.println("java ListAndGet <URL сервиса> <папка счетов-фактур>");

	}

	/**
	 * Запись файла
	 */
	private static void writeFile(File file, byte[] data) throws IOException {
		DataOutputStream os = new DataOutputStream(new FileOutputStream(file));
		try {
			os.write(data);
		} finally {
			os.close();
		}
	}

	/**
	 * Разбор форматированной даты и времени в формате 2016-11-05T13:15:30Z
	 */

	private static final SimpleDateFormat sdf;
	static {
		sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
		sdf.setTimeZone(TimeZone.getTimeZone("Europe/Minsk"));
	}

	public static Date string2Date(String date) throws ParseException {
		return sdf.parse(date);
	}

	/**
	 * Форматирование даты и времени к виду 2016-11-05T13:15:30Z
	 */
	public static String date2String(Date date) {
		return sdf.format(date);
	}

}
