package by.avest.edoc.examples;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.Security;

import javax.management.modelmbean.XMLParseException;

import by.avest.certstore.AvCertStoreProvider;
import by.avest.crypto.pkcs11.provider.AvestProvider;
import by.avest.crypto.pkcs11.provider.ProviderFactory;
import by.avest.edoc.client.AvEDoc;
import by.avest.edoc.client.AvETicket;
import by.avest.edoc.client.AvError;
import by.avest.edoc.client.EVatService;
import by.avest.net.tls.AvTLSProvider;

/**
 * Подпись и отправка счет-фактуры НДС на портал
 */
public class SignAndSend {

	private static final String XSD_FOR_ORIGINAL_TYPE = "MNSATI_original.xsd ";
	private static final String XSD_FOR_FIXED_TYPE = "MNSATI_fixed.xsd ";
	private static final String XSD_FOR_ADDITIONAL_TYPE = "MNSATI_additional.xsd ";

	public static void main(String[] args) {
		boolean isVerbose = true;

		AvestProvider prov = null;

		try {
			// Регистрация провайдера AvJceProv
			prov = ProviderFactory.addAvUniversalProvider();
			Security.addProvider(new AvTLSProvider());
			Security.addProvider(new AvCertStoreProvider());

			// Получение параметров командной строки
			if (args == null || args.length < 4) {
				printUsage("Ошибка: не указаны входные параметры");
				return;
			}

			String serviceUrl = args[0];
			String xsdFolderName = args[1];
			String archiveFolderName = args[2];
			String documentFileName = args[3];

			if (4 < args.length && args[4].equalsIgnoreCase("v")) {
				isVerbose = true;
			}

			// Создание экземпляра класса доступа к порталу
			EVatService service = null;

			try {
				service = new EVatService(serviceUrl,
						new KeyInteractiveSelector());

				// Авторизация, выбор ключа и ввода пароля выполняется
				// интерактивно
				// при помощи класса KeyInteractiveSelector
				service.login();

				// Подключение к автоматизированному сервису портала
				service.connect();

				// Создание электронного документа
				AvEDoc eDoc = service.createEDoc();

				// Загразка электронной счет-фактуры НДС
				eDoc.getDocument().load(readFile(new File(documentFileName)));

				String docUniqNumber = eDoc.getDocument().getXmlNodeValue(
						"issuance/general/number");
				String docType = eDoc.getDocument().getXmlNodeValue(
						"issuance/general/documentType");

				System.out.println("Номер счет-фактуры НДС: " + docUniqNumber);
				System.out.println("Тип счет-фактуры НДС: " + docType);

				// Проверка счет-фактуры НДС на соответствие XSD схеме
				byte[] xsdSchema = loadXsdSchema(xsdFolderName, docType);
				boolean isDocumentValid = eDoc.getDocument().validateXML(
						xsdSchema);

				if (!isDocumentValid) {
					System.err
							.println("Ошибка: структура документа не отвечает XSD схеме");
					System.exit(1);
				} else {
					System.out.println("Документ соответствует XSD схеме");
				}

				// Выработка ЭЦП
				eDoc.sign();

				byte[] signedDocument = eDoc.getEncoded();

				File signedFile = new File(archiveFolderName, new File(
						documentFileName).getName() + ".sgn.xml");

				// Получение параметров подписанного документа
				String signingTime = eDoc.getSignProperty(0,
						AvEDoc.PROP_SIGN_DATE);
				System.out.println("Время генерации подписи: " + signingTime);

				String organizationName = eDoc.getSignProperty(0, "2.5.4.10");
				String surname = eDoc.getSignProperty(0, "2.5.4.4");
				String name = eDoc.getSignProperty(0, "2.5.4.41");
				System.out
						.println("Информация о подписавшей стороне: организация '"
								+ organizationName
								+ "'; ответственное лицо '"
								+ surname + " " + name + "'.");

				// Сохранение файла с подписанным электронным документом
				writeFile(signedFile, signedDocument);

				// Загрузка электронного документа на автоматизированный сервис
				// портала и получение квитанции о приёме
				AvETicket ticket = service.sendEDoc(eDoc);
				System.out.println("Ответ сервера получен.");

				// Проверка квитанции
				if (ticket.accepted()) {
					String resultMessage = ticket.getMessage();

					File ticketFile = new File(archiveFolderName, new File(
							documentFileName).getName() + ".ticket.xml");
					// Сохранение квитанции в файл
					writeFile(ticketFile, ticket.getEncoded());

					System.out
							.println("Ответ сервера проверен. Cчет/фактура принята в обработку. "
									+ "Сообщение сервера: " + resultMessage);

				} else {
					AvError err = ticket.getLastError();

					File ticketFile = new File(archiveFolderName, new File(
							documentFileName).getName() + ".ticket.error.xml");
					// Сохранение квитанции в файл
					writeFile(ticketFile, ticket.getEncoded());

					System.err.println(err.getMessage());
					System.exit(1);
				}
			} catch (Exception e) {
				System.err.println("Ошибка: " + e.getMessage());
				if (isVerbose) {
					e.printStackTrace();
				}
				System.exit(1);
			} finally {
				if (service != null) {
					try {
						// Завершение работы с сервисом
						service.disconnect();

						// Завершение авторизованной сессии
						service.logout();
					} catch (IOException e) {
						// ignore
					}
				}
			}
		} finally {
			if (prov != null) {
				prov.close();
			}
		}
	}

	/**
	 * Отображение справки по параметрам командной строки
	 */
	private static void printUsage(String message) {
		System.out.println("SignAndSend: " + message);
		System.out.println("Использование:");
		System.out
				.println("java SignAndSend <URL сервиса> <папка XSD> <папка архива> <файл счет-фактуры> ");

	}

	/**
	 * Загрузка XSD схемы из файла
	 */
	private static byte[] loadXsdSchema(String xsdFolderName, String doctype)
			throws Exception {
		// validate XSD schema
		File xsdFile = null;
		doctype = (doctype == null) ? "" : doctype;

		if (doctype.equalsIgnoreCase("ORIGINAL")
				|| doctype.equalsIgnoreCase("ADD_NO_REFERENCE")) {
			xsdFile = new File(xsdFolderName, XSD_FOR_ORIGINAL_TYPE);
		} else if (doctype.equalsIgnoreCase("FIXED")) {
			xsdFile = new File(xsdFolderName, XSD_FOR_FIXED_TYPE);
		} else if (doctype.equalsIgnoreCase("ADDITIONAL")) {
			xsdFile = new File(xsdFolderName, XSD_FOR_ADDITIONAL_TYPE);
		} else {
			throw new XMLParseException(
					"Ошибка: неизвестный тип счет-фактуры '" + doctype + "'.");
		}

		if (!xsdFile.exists() && !xsdFile.isFile()) {
			throw new Exception("Ошибка: невозможно загрузить XSD файл \""
					+ xsdFile.getAbsolutePath() + "\"");
		}

		byte[] result = readFile(xsdFile);

		return result;
	}

	/**
	 * Чтение файла
	 */
	private static byte[] readFile(File file) throws FileNotFoundException,
			IOException {
		byte[] fileData = new byte[(int) file.length()];
		DataInputStream dis = new DataInputStream(new FileInputStream(file));
		try {
			dis.readFully(fileData);
		} finally {
			dis.close();
		}
		return fileData;
	}

	/**
	 * Запись файла
	 */
	private static void writeFile(File file, byte[] data) throws IOException {
		DataOutputStream os = new DataOutputStream(new FileOutputStream(file));
		try {
			os.write(data);
		} finally {
			os.close();
		}
	}
}
