package by.avest.edoc.examples;

import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.Security;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

import by.avest.certstore.AvCertStoreProvider;
import by.avest.crypto.pkcs11.provider.AvestProvider;
import by.avest.crypto.pkcs11.provider.ProviderFactory;
import by.avest.edoc.client.AvEStatus;
import by.avest.edoc.client.EVatService;
import by.avest.net.tls.AvTLSProvider;

/**
 * Получение статуса обработки счета-фактуры НДС с портала
 */
public class GetStatus {

	public static void main(String[] args) {
		boolean isVerbose = false;

		AvestProvider prov = null;

		try {
			// Регистрация провайдера AvJceProv
			prov = ProviderFactory.addAvUniversalProvider();
			Security.addProvider(new AvTLSProvider());
			Security.addProvider(new AvCertStoreProvider());

			// Получение параметров командной строки
			if (args.length < 3) {
				printUsage("Ошибка: не указаны входные параметры");
				return;
			}

			String serviceUrl = args[0];
			String docFolderName = args[1];
			String invoiceNum = args[2];

			if (3 < args.length && args[3].equalsIgnoreCase("v")) {
				isVerbose = true;
			}

			// Создание экземпляра класса доступа к порталу
			EVatService service = null;
			try {
				service = new EVatService(serviceUrl, new KeyInteractiveSelector());

				// Авторизация, выбор ключа и ввода пароля выполняется
				// интерактивно
				// при помощи класса KeyInteractiveSelector
				service.login();

				// Подключение к автоматизированному сервису портала
				service.connect();

				// Получение списка поступивщих счетов-фактур НДС
				AvEStatus documentStatus = service.getStatus(invoiceNum);

				System.out.println("Получен статус счета-фактуры \"" + invoiceNum + "\"");

				// Проверка ЭЦП электронного документа
				boolean isValid = documentStatus.verify();
				if (isValid) {
					String dateStr = prepareFileDate(new Date());

					// Сохранение электронного документа "Статус" в файл
					File docFile = new File(docFolderName, "invoice" + invoiceNum + ".xml." + dateStr + ".status.xml");
					writeFile(docFile, documentStatus.getEncoded());

					System.out.println("Статус:  " + documentStatus.getStatus());

					System.out.println("Сообщение:  " + documentStatus.getMessage());

					System.out.println("Дата изменения статуса:  " + documentStatus.getSince());

				} else {

					System.err.println(
							"Ошибка: получен некорректный статус: " + documentStatus.getLastError().getMessage());

				}
			} catch (Exception e) {
				System.err.println("Ошибка: " + e.getMessage());
				if (isVerbose) {
					e.printStackTrace();
				}
				System.exit(1);
			} finally {
				if (service != null) {
					try {
						// Завершение работы с сервисом
						service.disconnect();

						// Завершение авторизованной сессии
						service.logout();
					} catch (IOException e) {
						// ignore
					}
				}
			}
		} finally {
			if (prov != null) {
				prov.close();
			}
		}
	}

	/**
	 * Отображение справки по параметрам командной строки
	 */
	private static void printUsage(String message) {
		System.out.println("GetStatus: " + message);
		System.out.println("Использование:");
		System.out.println("java GetStatus <URL сервиса> <папка сохранения> <номер счета-фактуры>");

	}

	/**
	 * Запись файла
	 */
	private static void writeFile(File file, byte[] data) throws IOException {
		DataOutputStream os = new DataOutputStream(new FileOutputStream(file));
		try {
			os.write(data);
		} finally {
			os.close();
		}
	}

	/**
	 * Форматирование даты и
	 */
	private static String prepareFileDate(Date date) {
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy_MM_dd");
		sdf.setTimeZone(TimeZone.getTimeZone("Europe/Minsk"));
		return sdf.format(date);
	}

}
