package by.avest.edoc.examples;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.Security;

import by.avest.certstore.AvCertStoreProvider;
import by.avest.crypto.pkcs11.provider.AvestProvider;
import by.avest.crypto.pkcs11.provider.ProviderFactory;
import by.avest.edoc.client.AvDocException;
import by.avest.edoc.client.AvEDoc;
import by.avest.edoc.client.AvETicket;
import by.avest.edoc.client.AvError;
import by.avest.edoc.client.EVatService;
import by.avest.net.tls.AvTLSProvider;

/**
 * Добавление подписи в подписанный счет-фактуру НДС и загрузка на портал
 * 
 */
public class VerifySignSend {

	public static void main(String[] args) {
		boolean isVerbose = false;

		AvestProvider prov = null;

		try {
			// Регистрация провайдера AvJceProv
			prov = ProviderFactory.addAvUniversalProvider();
			Security.addProvider(new AvTLSProvider());
			Security.addProvider(new AvCertStoreProvider());

			// Получение параметров командной строки
			if (args.length < 3) {
				printUsage("Ошибка: не указаны входные параметры");
				return;
			}

			String serviceUrl = args[0];
			String archiveFolderName = args[1];
			String documentFileName = args[2];

			if (3 < args.length && args[3].equalsIgnoreCase("v")) {
				isVerbose = true;
			}

			// Создание экземпляра класса доступа к порталу
			EVatService service = null;

			try {
				service = new EVatService(serviceUrl, new KeyInteractiveSelector());

				// Авторизация, выбор ключа и ввода пароля выполняется
				// интерактивно
				// при помощи класса KeyInteractiveSelector
				service.login();

				// Подключение к автоматизированному сервису портала
				service.connect();

				byte[] data = readFile(new File(documentFileName));

				// load
				AvEDoc eDocXml = service.createEDoc();
				eDocXml.load(data);

				String docNumber = eDocXml.getDocument().getXmlNodeValue("issuance/general/number");

				// verify
				boolean isValid = edocVerify(eDocXml);
				if (!isValid) {
					System.err.println("Ошибка проверки файла со счетом-фактурой НДС под номером " + docNumber + ": "
							+ eDocXml.getLastError().getMessage());
					return;
				}

				// Выработка ЭЦП
				eDocXml.sign();

				byte[] signedDocument = eDocXml.getEncoded();

				File signedFile = new File(archiveFolderName, new File(documentFileName).getName() + ".signed.xml");

				// Сохранение файла с подписанным электронным документом
				writeFile(signedFile, signedDocument);

				// Загрузка электронного документа на автоматизированный сервис
				// портала и получение квитанции о приёме
				AvETicket ticket = service.sendEDoc(eDocXml);

				File ticketFile = new File(archiveFolderName, new File(documentFileName).getName() + ".ticket.xml");
				// Сохранение квитанции в файл
				writeFile(ticketFile, ticket.getEncoded());

				// Проверка квитанции
				if (ticket.accepted()) {
					String resultMessage = ticket.getMessage();

					System.out.println("Счет-фактура НДС принят к обработке: " + resultMessage);

				} else {
					AvError err = ticket.getLastError();
					System.err.println("Ошибка: счет-фактура не принят. Сообщение: " + err.getMessage() + ".");
					System.exit(1);
				}
			} catch (Exception e) {
				System.err.println("Ошибка: " + e.getMessage());
				if (isVerbose) {
					e.printStackTrace();
				}
				System.exit(1);
			} finally {
				if (service != null) {
					try {
						// Завершение работы с сервисом
						service.disconnect();

						// Завершение авторизованной сессии
						service.logout();
					} catch (IOException e) {
						// ignore
					}
				}
			}
		} finally {
			if (prov != null) {
				prov.close();
			}
		}
	}

	protected static final String OID_CN = "2.5.4.3";

	private static final String SERVER_DN_CN = "Автоматизированный сервис портала АИС УСФ";

	protected static boolean isServerSign(AvEDoc edoc, int i) throws IOException, AvDocException {
		String commonName = edoc.getSignProperty(i, OID_CN);

		return (SERVER_DN_CN.equals(commonName));
	}

	private static int findServerSign(AvEDoc edoc) throws IOException, AvDocException {
		int result = -1;

		int signCount = edoc.getSignCount();

		for (int i = 0; i < signCount; i++) {
			if (isServerSign(edoc, i)) {
				// Найдена подпись сервера
				result = i;
				break;
			}
		}

		return result;
	}

	private static boolean edocVerify(AvEDoc edoc) throws AvDocException, IOException {
		boolean isValid = false;

		// Количество подписей
		int signCount = edoc.getSignCount();
		if (signCount == 0) {
			System.out.println("Документ не содержит ЭЦП");
			return false;
		}

		// Найти подпись сервера
		int serverIndex = findServerSign(edoc);

		if (0 < serverIndex) {
			System.out.println("Проверка подписи сервера");

			isValid = edoc.verifySign(serverIndex);

			// Print verification result
			if (isValid) {
				String signdate = edoc.getSignProperty(serverIndex, AvEDoc.PROP_SIGN_DATE);
				System.out.println("ЭЦП проверена. Дата подписания: " + signdate);
			} else {
				System.out.println(
						"ЭЦП неверна. " + ((edoc.getLastError() != null) ? edoc.getLastError().getMessage() : "-"));
			}
		} else {
			System.out.println("Подпись сервера не найдена. Будет выполнена проверка всех подписей.");

			int count = edoc.getSignCount();

			for (int j = 0; j < count; j++) {
				// Информация о стороне подписавшей документ
				String cn = edoc.getSignProperty(j, "2.5.4.3");
				System.out.println("\tПроверка подписи " + cn);

				// Проверка ЭЦП электронного документа
				isValid = edoc.verifySign(j);

				if (!isValid) {
					break;
				}
			}

			// Print verification result
			if (isValid) {
				System.out.println("Все ЭЦП проверены успешно.");
			} else {
				System.out.println(
						"ЭЦП неверна. " + ((edoc.getLastError() != null) ? edoc.getLastError().getMessage() : "-"));
			}
		}

		return isValid;
	}

	/**
	 * Отображение справки по параметрам командной строки
	 */
	private static void printUsage(String message) {
		System.out.println("VerifySignSend: " + message);
		System.out.println("Использование:");
		System.out.println("java VerifySignSend <URL сервиса> <папка архива> <файл счета-фактуры> ");

	}

	/**
	 * Чтение файла
	 */
	private static byte[] readFile(File file) throws FileNotFoundException, IOException {
		byte[] fileData = new byte[(int) file.length()];
		DataInputStream dis = new DataInputStream(new FileInputStream(file));
		try {
			dis.readFully(fileData);
		} finally {
			dis.close();
		}
		return fileData;
	}

	/**
	 * Запись файла
	 */
	private static void writeFile(File file, byte[] data) throws IOException {
		DataOutputStream os = new DataOutputStream(new FileOutputStream(file));
		try {
			os.write(data);
		} finally {
			os.close();
		}
	}
}
